<?php
/**
 * User Login Endpoint - DEBUG VERSION
 * POST /api/auth/login-debug
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

// Set headers
setCorsHeaders();
setCacheNone();
setJsonHeader();

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Method not allowed', 405);
}

try {
    // Get request body
    $data = getJsonBody();

    error_log('[Login-Debug] Request data: ' . json_encode($data));

    // Validate required parameters
    $validation = validateParams(['username', 'password'], $data);
    if ($validation !== true) {
        error_log('[Login-Debug] Validation failed: ' . $validation);
        sendError($validation, 400);
    }

    $username = sanitizeInput($data['username']);
    $password = $data['password'];

    error_log('[Login-Debug] Username: ' . $username);
    error_log('[Login-Debug] Password length: ' . strlen($password));

    // Hash password for comparison
    $passwordHash = hashPassword($password);
    error_log('[Login-Debug] Password hash: ' . $passwordHash);

    // First, check if user exists
    $checkSql = "SELECT id, username, password_hash FROM users WHERE username = :username";
    $checkResult = dbQuery($checkSql, ['username' => $username]);

    error_log('[Login-Debug] User query result count: ' . count($checkResult));

    if (empty($checkResult)) {
        error_log('[Login-Debug] User not found');
        sendError('User not found', 401);
    }

    $userCheck = $checkResult[0];
    error_log('[Login-Debug] Stored password hash: ' . $userCheck['password_hash']);
    error_log('[Login-Debug] Hashes match: ' . ($userCheck['password_hash'] === $passwordHash ? 'YES' : 'NO'));

    // Query user with premium info
    $sql = "
        SELECT
            id, username, created_at, last_login, status,
            premium_status, premium_expires_at, premium_started_at, plan_type
        FROM users
        WHERE username = :username AND password_hash = :password_hash
    ";

    $result = dbQuery($sql, [
        'username' => $username,
        'password_hash' => $passwordHash
    ]);

    // Check if user exists
    if (empty($result)) {
        error_log('[Login-Debug] Login failed - invalid credentials');
        sendError('Invalid username or password', 401);
    }

    $user = $result[0];
    error_log('[Login-Debug] User found: ' . json_encode($user));

    // Check account status
    if ($user['status'] !== 'active') {
        error_log('[Login-Debug] Account disabled');
        sendError('Account is disabled', 403);
    }

    // Update last login
    try {
        $updateSql = "UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = :id";
        dbExecute($updateSql, ['id' => $user['id']]);
        error_log('[Login-Debug] Last login updated');
    } catch (Exception $e) {
        error_log('[Login-Debug] Could not update last_login: ' . $e->getMessage());
    }

    // Generate session token
    $sessionToken = generateSessionToken();
    error_log('[Login-Debug] Session token generated: ' . $sessionToken);

    // Success response with user data and session
    sendSuccess([
        'user' => [
            'id' => (int)$user['id'],
            'username' => $user['username'],
            'created_at' => $user['created_at'],
            'premium_status' => $user['premium_status'] ?? 'free',
            'premium_expires_at' => $user['premium_expires_at'],
            'premium_started_at' => $user['premium_started_at'],
            'plan_type' => $user['plan_type'] ?? 'free',
        ],
        'session_token' => $sessionToken
    ], 'Login successful');

} catch (Exception $e) {
    error_log('[Login-Debug] Exception: ' . $e->getMessage());
    error_log('[Login-Debug] Stack trace: ' . $e->getTraceAsString());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
