<?php
/**
 * Fast Login Endpoint (No session table, no last_login update)
 * POST /api/auth/login-fast
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

// Set headers
setCorsHeaders();
setCacheNone();
setJsonHeader();

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Method not allowed', 405);
}

try {
    $startTime = microtime(true);

    // Get request body
    $data = getJsonBody();
    $parseTime = microtime(true) - $startTime;

    // Validate required parameters
    $validation = validateParams(['username', 'password'], $data);
    if ($validation !== true) {
        sendError($validation, 400);
    }

    $username = sanitizeInput($data['username']);
    $password = $data['password'];

    // Hash password for comparison
    $hashStart = microtime(true);
    $passwordHash = hashPassword($password);
    $hashTime = microtime(true) - $hashStart;

    // Query user with premium info (READ ONLY - No writes!)
    $queryStart = microtime(true);
    $sql = "
        SELECT
            id, username, created_at, last_login, status,
            premium_status, premium_expires_at, premium_started_at, plan_type
        FROM users
        WHERE username = :username AND password_hash = :password_hash
        LIMIT 1
    ";

    $result = dbQuery($sql, [
        'username' => $username,
        'password_hash' => $passwordHash
    ]);
    $queryTime = microtime(true) - $queryStart;

    // Check if user exists
    if (empty($result)) {
        $totalTime = microtime(true) - $startTime;
        sendError('Invalid username or password (took ' . round($totalTime * 1000, 2) . 'ms)', 401);
    }

    $user = $result[0];

    // Check account status
    if ($user['status'] !== 'active') {
        sendError('Account is disabled', 403);
    }

    // Generate session token (but don't save to database)
    $tokenStart = microtime(true);
    $sessionToken = generateSessionToken();
    $tokenTime = microtime(true) - $tokenStart;

    $totalTime = microtime(true) - $startTime;

    // Success response with user data and session
    sendSuccess([
        'user' => [
            'id' => (int)$user['id'],
            'username' => $user['username'],
            'created_at' => $user['created_at'],
            'premium_status' => $user['premium_status'] ?? 'free',
            'premium_expires_at' => $user['premium_expires_at'],
            'premium_started_at' => $user['premium_started_at'],
            'plan_type' => $user['plan_type'] ?? 'free',
        ],
        'session_token' => $sessionToken,
        'debug_timing' => [
            'parse_time' => round($parseTime * 1000, 2) . 'ms',
            'hash_time' => round($hashTime * 1000, 2) . 'ms',
            'query_time' => round($queryTime * 1000, 2) . 'ms',
            'token_time' => round($tokenTime * 1000, 2) . 'ms',
            'total_time' => round($totalTime * 1000, 2) . 'ms'
        ]
    ], 'Fast login successful');

} catch (Exception $e) {
    error_log('[LoginFast] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
