<?php
/**
 * Production Login Endpoint
 * POST /api/auth/login-final
 *
 * OPTIMIZED: Checks password in PHP instead of SQL
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

setCorsHeaders();
setCacheNone();
setJsonHeader();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Method not allowed', 405);
}

try {
    $startTime = microtime(true);

    // Get request body
    $data = getJsonBody();

    // Validate required parameters
    $validation = validateParams(['username', 'password'], $data);
    if ($validation !== true) {
        sendError($validation, 400);
    }

    $username = sanitizeInput($data['username']);
    $password = $data['password'];

    // STEP 1: Find user by username only (fast query with index)
    $queryStart = microtime(true);
    $sql = "
        SELECT
            id, username, password_hash, created_at, last_login, status,
            premium_status, premium_expires_at, premium_started_at, plan_type
        FROM users
        WHERE username = :username
        LIMIT 1
    ";

    $result = dbQuery($sql, ['username' => $username]);
    $queryTime = microtime(true) - $queryStart;

    // Check if user exists
    if (empty($result)) {
        error_log('[LoginFinal] User not found: ' . $username);
        sendError('Invalid username or password', 401);
    }

    $user = $result[0];

    // STEP 2: Verify password in PHP (not in SQL!)
    $hashStart = microtime(true);
    $passwordHash = hashPassword($password);
    $hashTime = microtime(true) - $hashStart;

    $verifyStart = microtime(true);
    if ($user['password_hash'] !== $passwordHash) {
        $verifyTime = microtime(true) - $verifyStart;
        error_log('[LoginFinal] Invalid password for user: ' . $username . ' (verify took ' . round($verifyTime * 1000, 2) . 'ms)');
        sendError('Invalid username or password', 401);
    }
    $verifyTime = microtime(true) - $verifyStart;

    // Check account status
    if ($user['status'] !== 'active') {
        sendError('Account is disabled', 403);
    }

    // STEP 3: Update last login (optional, can fail silently)
    try {
        $updateSql = "UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = :id";
        dbExecute($updateSql, ['id' => $user['id']]);
    } catch (Exception $e) {
        error_log('[LoginFinal] Could not update last_login: ' . $e->getMessage());
    }

    // STEP 4: Generate session token
    $tokenStart = microtime(true);
    $sessionToken = generateSessionToken();
    $tokenTime = microtime(true) - $tokenStart;

    // STEP 5: Try to save session (optional)
    try {
        $sessionSql = "
            INSERT INTO user_sessions (user_id, session_token, expires_at)
            VALUES (:user_id, :token, CURRENT_TIMESTAMP + INTERVAL '30 days')
        ";
        dbExecute($sessionSql, [
            'user_id' => $user['id'],
            'token' => $sessionToken
        ]);
    } catch (Exception $e) {
        error_log('[LoginFinal] Could not save session: ' . $e->getMessage());
    }

    $totalTime = microtime(true) - $startTime;

    // Success response
    sendSuccess([
        'user' => [
            'id' => (int)$user['id'],
            'username' => $user['username'],
            'created_at' => $user['created_at'],
            'premium_status' => $user['premium_status'] ?? 'free',
            'premium_expires_at' => $user['premium_expires_at'],
            'premium_started_at' => $user['premium_started_at'],
            'plan_type' => $user['plan_type'] ?? 'free',
        ],
        'session_token' => $sessionToken,
        'debug_timing' => [
            'query_time' => round($queryTime * 1000, 2) . 'ms',
            'hash_time' => round($hashTime * 1000, 2) . 'ms',
            'verify_time' => round($verifyTime * 1000, 2) . 'ms',
            'token_time' => round($tokenTime * 1000, 2) . 'ms',
            'total_time' => round($totalTime * 1000, 2) . 'ms'
        ]
    ], 'Login successful');

} catch (Exception $e) {
    error_log('[LoginFinal] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
