<?php
/**
 * User Login Endpoint
 * POST /api/auth/login
 *
 * Request Body:
 * {
 *   "username": "user123",
 *   "password": "password123"
 * }
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

// Set headers
setCorsHeaders();
setCacheNone(); // Never cache authentication
setJsonHeader();

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Method not allowed', 405);
}

try {
    // Get request body
    $data = getJsonBody();

    // Validate required parameters
    $validation = validateParams(['username', 'password'], $data);
    if ($validation !== true) {
        sendError($validation, 400);
    }

    $username = sanitizeInput($data['username']);
    $password = $data['password'];

    // OPTIMIZED: Query user by username only (fast with index)
    // Then verify password in PHP (not in SQL WHERE clause)
    $sql = "
        SELECT
            id, username, password_hash, created_at, last_login, status,
            premium_status, premium_expires_at, premium_started_at, plan_type
        FROM users
        WHERE username = :username
        LIMIT 1
    ";

    $result = dbQuery($sql, ['username' => $username]);

    // Check if user exists
    if (empty($result)) {
        sendError('Invalid username or password', 401);
    }

    $user = $result[0];

    // Verify password in PHP (not in SQL!)
    $passwordHash = hashPassword($password);
    if ($user['password_hash'] !== $passwordHash) {
        sendError('Invalid username or password', 401);
    }

    // Check account status
    if ($user['status'] !== 'active') {
        sendError('Account is disabled', 403);
    }

    // Update last login
    $updateSql = "UPDATE users SET last_login = CURRENT_TIMESTAMP WHERE id = :id";
    dbExecute($updateSql, ['id' => $user['id']]);

    // Generate session token
    $sessionToken = generateSessionToken();

    // Try to create session (if table exists)
    try {
        $sessionSql = "
            INSERT INTO user_sessions (user_id, session_token, expires_at)
            VALUES (:user_id, :token, CURRENT_TIMESTAMP + INTERVAL '30 days')
        ";
        dbExecute($sessionSql, [
            'user_id' => $user['id'],
            'token' => $sessionToken
        ]);
    } catch (Exception $e) {
        // Ignore if user_sessions table doesn't exist yet
        error_log('[Login] Could not save session (table may not exist): ' . $e->getMessage());
    }

    // Success response with user data and session
    sendSuccess([
        'user' => [
            'id' => (int)$user['id'],
            'username' => $user['username'],
            'created_at' => $user['created_at'],
            'premium_status' => $user['premium_status'] ?? 'free',
            'premium_expires_at' => $user['premium_expires_at'],
            'premium_started_at' => $user['premium_started_at'],
            'plan_type' => $user['plan_type'] ?? 'free',
        ],
        'session_token' => $sessionToken
    ], 'Login successful');

} catch (Exception $e) {
    error_log('[Login] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
