<?php
/**
 * User Registration Endpoint
 * POST /api/auth/register
 *
 * Request Body:
 * {
 *   "username": "user123",
 *   "password": "password123"
 * }
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

// Set headers
setCorsHeaders();
setCacheNone(); // Never cache authentication
setJsonHeader();

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Method not allowed', 405);
}

try {
    // Get request body
    $data = getJsonBody();

    // Validate required parameters
    $validation = validateParams(['username', 'password'], $data);
    if ($validation !== true) {
        sendError($validation, 400);
    }

    $username = sanitizeInput($data['username']);
    $password = $data['password'];

    // Validation
    if (strlen($username) < 3) {
        sendError('Username must be at least 3 characters', 400);
    }

    if (strlen($password) < 6) {
        sendError('Password must be at least 6 characters', 400);
    }

    // Check if username already exists
    $checkSql = "SELECT id FROM users WHERE username = :username";
    $existing = dbQuery($checkSql, ['username' => $username]);

    if (!empty($existing)) {
        sendError('Username already exists', 409);
    }

    // Hash password
    $passwordHash = hashPassword($password);

    // Insert new user
    $insertSql = "
        INSERT INTO users (username, password_hash, created_at, updated_at, status)
        VALUES (:username, :password_hash, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, 'active')
        RETURNING id, username, created_at
    ";

    $result = dbQuery($insertSql, [
        'username' => $username,
        'password_hash' => $passwordHash
    ]);

    if (empty($result)) {
        sendError('Registration failed', 500);
    }

    $user = $result[0];

    // Success response
    sendSuccess([
        'user' => [
            'id' => (int)$user['id'],
            'username' => $user['username'],
            'created_at' => $user['created_at']
        ]
    ], 'Registration successful');

} catch (Exception $e) {
    error_log('[Register] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
