<?php
/**
 * Detail Endpoint - TIMING DEBUG
 * VERSION: 2024-10-17-TIMING-DEBUG
 */

function logTime($label) {
    global $timings;
    $timings[] = [
        'label' => $label,
        'time' => microtime(true),
        'memory' => memory_get_usage()
    ];
    error_log("[TIMING] $label: " . microtime(true));
}

$timings = [];
$startTime = microtime(true);
logTime('START');

require_once '../../../spam/db_config.php';
logTime('AFTER require db_config.php');

require_once '../../../spam/cache_config.php';
logTime('AFTER require cache_config.php');

require_once '../../../spam/helpers.php';
logTime('AFTER require helpers.php');

setCorsHeaders();
logTime('AFTER setCorsHeaders()');

setCacheDynamic(240);
logTime('AFTER setCacheDynamic()');

setJsonHeader();
logTime('AFTER setJsonHeader()');

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Method not allowed', 405);
}
logTime('AFTER method check');

try {
    $id = (int)getParam('id', 123);
    logTime('AFTER getParam id');

    if ($id <= 0) {
        sendError('Invalid content ID', 400);
    }
    logTime('AFTER id validation');

    $sql = "
        SELECT id, title, description, cover_url, rating,
               source_site, year, genres, alternative_titles,
               created_at, updated_at
        FROM animes
        WHERE id = :id
        LIMIT 1
    ";
    logTime('AFTER SQL string build');

    // ⏱️ CRITICAL: Measure getDbConnection time
    $pdo = getDbConnection(true);
    logTime('AFTER getDbConnection() - THIS IS CRITICAL!');

    // ⏱️ CRITICAL: Measure prepare time
    $stmt = $pdo->prepare($sql);
    logTime('AFTER prepare()');

    // ⏱️ CRITICAL: Measure execute time
    $stmt->execute(['id' => $id]);
    logTime('AFTER execute()');

    // ⏱️ CRITICAL: Measure fetchAll time
    $results = $stmt->fetchAll();
    logTime('AFTER fetchAll() - rows: ' . count($results));

    if (empty($results)) {
        sendError('Content not found', 404);
    }
    logTime('AFTER empty check');

    // ⏱️ Measure contentFromRow time
    $content = contentFromRow($results[0]);
    logTime('AFTER contentFromRow()');

    // Calculate detailed timings
    $processedTimings = [];
    for ($i = 1; $i < count($timings); $i++) {
        $prev = $timings[$i - 1];
        $curr = $timings[$i];
        $duration = ($curr['time'] - $prev['time']) * 1000;
        $processedTimings[] = [
            'step' => $curr['label'],
            'duration_ms' => round($duration, 3),
            'memory_mb' => round($curr['memory'] / 1024 / 1024, 2)
        ];
    }

    $totalTime = round((microtime(true) - $startTime) * 1000, 2);

    sendSuccess([
        'content' => $content,
        '_debug' => [
            'version' => '2024-10-17-TIMING-DEBUG',
            'total_time_ms' => $totalTime,
            'timings' => $processedTimings
        ]
    ]);

} catch (Exception $e) {
    logTime('EXCEPTION: ' . $e->getMessage());

    $processedTimings = [];
    for ($i = 1; $i < count($timings); $i++) {
        $prev = $timings[$i - 1];
        $curr = $timings[$i];
        $duration = ($curr['time'] - $prev['time']) * 1000;
        $processedTimings[] = [
            'step' => $curr['label'],
            'duration_ms' => round($duration, 3)
        ];
    }

    error_log('[Detail Timing] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage() . ' | Timings: ' . json_encode($processedTimings), 500);
}
?>
