<?php
/**
 * Debug version of movies.php with timing
 */

$scriptStart = microtime(true);
$timings = [];

$step = microtime(true);
require_once '../../../spam/db_config.php';
$timings['require_db_config'] = (microtime(true) - $step) * 1000;

$step = microtime(true);
require_once '../../../spam/cache_config.php';
$timings['require_cache_config'] = (microtime(true) - $step) * 1000;

$step = microtime(true);
require_once '../../../spam/helpers.php';
$timings['require_helpers'] = (microtime(true) - $step) * 1000;

$step = microtime(true);
setCorsHeaders();
setCacheLong();
setJsonHeader();
$timings['set_headers'] = (microtime(true) - $step) * 1000;

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Method not allowed', 405);
}

try {
    $step = microtime(true);
    $language = getParam('language', 'de');
    $limit = min((int)getParam('limit', 20), 100);
    $offset = max((int)getParam('offset', 0), 0);
    $orderBy = getParam('orderBy', 'updated_at');
    $timings['get_params'] = (microtime(true) - $step) * 1000;

    $allowedOrderBy = ['updated_at', 'created_at', 'rating', 'title'];
    if (!in_array($orderBy, $allowedOrderBy)) {
        $orderBy = 'updated_at';
    }

    $step = microtime(true);
    $sourceSites = getMovieSitesByLanguage($language);
    $sourceSitesStr = "'" . implode("','", $sourceSites) . "'";
    $timings['get_source_sites'] = (microtime(true) - $step) * 1000;

    $step = microtime(true);
    $sql = "
        SELECT
            id, title, description, cover_url, rating,
            source_site, year, genres, alternative_titles,
            created_at, updated_at
        FROM animes
        WHERE source_site IN ($sourceSitesStr)
        AND status = 'completed'
        ORDER BY $orderBy DESC
        LIMIT :limit OFFSET :offset
    ";

    $results = dbQuery($sql, [
        'limit' => $limit,
        'offset' => $offset
    ]);
    $timings['db_query'] = (microtime(true) - $step) * 1000;
    $timings['result_count'] = count($results);

    $step = microtime(true);
    $movies = array_map('contentFromRow', $results);
    $timings['array_map'] = (microtime(true) - $step) * 1000;

    $timings['total_before_response'] = (microtime(true) - $scriptStart) * 1000;

    header('Content-Type: application/json');
    echo json_encode([
        'success' => true,
        'data' => [
            'items' => $movies,
            'count' => count($movies),
            'limit' => $limit,
            'offset' => $offset,
            'language' => $language
        ],
        'debug' => [
            'timings' => $timings,
            'total_ms' => round((microtime(true) - $scriptStart) * 1000, 3)
        ],
        'timestamp' => time()
    ]);

} catch (Exception $e) {
    error_log('[Movies Debug] Error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => ['timings' => $timings]
    ]);
}
?>
