<?php
/**
 * Get Series Endpoint
 * GET /api/content/series?language=de&limit=20&offset=0&orderBy=updated_at
 * VERSION: 2024-10-17-FIXED
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

// Set headers
setCorsHeaders();
setCacheLong(); // Cache for 2 hours
setJsonHeader();

// Only allow GET
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Method not allowed', 405);
}

try {
    // Get parameters
    $language = getParam('language', 'de');
    $limit = min((int)getParam('limit', 20), 100);
    $offset = max((int)getParam('offset', 0), 0);
    $orderBy = getParam('orderBy', 'updated_at');

    // Validate orderBy
    $allowedOrderBy = ['updated_at', 'created_at', 'rating', 'title'];
    if (!in_array($orderBy, $allowedOrderBy)) {
        $orderBy = 'updated_at';
    }

    // Get source sites for series
    $sourceSites = getSeriesSitesByLanguage($language);
    $sourceSitesStr = "'" . implode("','", $sourceSites) . "'";

    // Query series
    $sql = "
        SELECT
            id, title, description, cover_url, rating,
            source_site, year, genres, alternative_titles,
            created_at, updated_at
        FROM animes
        WHERE source_site IN ($sourceSitesStr)
        AND status = 'completed'
        ORDER BY $orderBy DESC
        LIMIT :limit OFFSET :offset
    ";

    // Bind parameters as integers for PostgreSQL
    $stmt = getDbConnection(true)->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $results = $stmt->fetchAll();

    // Convert to content objects
    $series = array_map('contentFromRow', $results);

    // Success response
    sendSuccess([
        'items' => $series,
        'count' => count($series),
        'limit' => $limit,
        'offset' => $offset,
        'language' => $language,
        '_version' => '2024-10-17-FIXED'
    ]);

} catch (Exception $e) {
    error_log('[Series] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
