<?php
/**
 * Test if LIMIT works in PostgreSQL query
 */

require_once '../../../spam/db_config.php';

header('Content-Type: application/json');

try {
    $pdo = getDbConnection(true); // PgBouncer

    // Test 1: Direct query with LIMIT (no parameters)
    $start = microtime(true);
    $stmt1 = $pdo->query("SELECT id FROM animes WHERE source_site = 'filmpalast' AND status = 'completed' ORDER BY updated_at DESC LIMIT 5");
    $result1 = $stmt1->fetchAll();
    $time1 = (microtime(true) - $start) * 1000;

    // Test 2: Prepared statement with LIMIT parameter (string binding)
    $start = microtime(true);
    $stmt2 = $pdo->prepare("SELECT id FROM animes WHERE source_site = 'filmpalast' AND status = 'completed' ORDER BY updated_at DESC LIMIT :limit");
    $stmt2->execute(['limit' => 5]);
    $result2 = $stmt2->fetchAll();
    $time2 = (microtime(true) - $start) * 1000;

    // Test 3: Prepared statement with LIMIT parameter (integer binding)
    $start = microtime(true);
    $stmt3 = $pdo->prepare("SELECT id FROM animes WHERE source_site = 'filmpalast' AND status = 'completed' ORDER BY updated_at DESC LIMIT :limit");
    $stmt3->bindValue(':limit', 5, PDO::PARAM_INT);
    $stmt3->execute();
    $result3 = $stmt3->fetchAll();
    $time3 = (microtime(true) - $start) * 1000;

    echo json_encode([
        'test1_direct_limit' => [
            'rows_returned' => count($result1),
            'time_ms' => round($time1, 2),
            'expected' => 5
        ],
        'test2_param_string' => [
            'rows_returned' => count($result2),
            'time_ms' => round($time2, 2),
            'expected' => 5
        ],
        'test3_param_int' => [
            'rows_returned' => count($result3),
            'time_ms' => round($time3, 2),
            'expected' => 5
        ],
        'conclusion' => count($result1) == 5 && count($result2) == 5 && count($result3) == 5
            ? 'LIMIT works correctly'
            : 'LIMIT is broken!'
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage()]);
}
?>
