<?php
/**
 * Get Episodes for Content
 * GET /api/episodes/episodes?content_id=123&season=1
 * USES: mv_episodes_with_streams (9 columns verified)
 * VERSION: 2024-10-17-OPCACHE-BUSTER
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

setCorsHeaders();
setCacheDynamic(360); // Cache for 6 hours
setJsonHeader();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Method not allowed', 405);
}

try {
    // Support both content_id and contentId for compatibility
    $contentId = (int)getParam('content_id', getParam('contentId', 0));
    $season = getParam('season');

    if ($contentId <= 0) {
        sendError('Invalid content_id', 400);
    }

    // Use materialized view (verified 9 columns)
    $sql = "
        SELECT
            mv.episode_id,
            mv.content_id,
            mv.episode_number,
            mv.title,
            mv.season_number,
            mv.season_id,
            mv.episode_language,
            mv.available_languages,
            mv.total_streams,
            st.language as stream_language,
            st.hoster,
            st.stream_url,
            st.quality
        FROM mv_episodes_with_streams mv
        LEFT JOIN streams st ON st.episode_id = mv.episode_id
        WHERE mv.content_id = :content_id
    ";

    $params = ['content_id' => $contentId];

    if ($season !== null && $season > 0) {
        $sql .= " AND mv.season_number = :season";
        $params['season'] = (int)$season;
    }

    $sql .= " ORDER BY mv.season_number, mv.episode_number, st.language, st.hoster";

    // Use PgBouncer for connection pooling (Materialized Views ARE accessible via PgBouncer)
    $results = dbQuery($sql, $params);

    // Group by episode
    $episodesMap = [];

    foreach ($results as $row) {
        $episodeId = (int)$row['episode_id'];

        if (!isset($episodesMap[$episodeId])) {
            // Decode available_languages from JSONB
            $availableLanguages = $row['available_languages'];
            if (is_string($availableLanguages)) {
                $availableLanguages = json_decode($availableLanguages, true);
            }
            if (!is_array($availableLanguages)) {
                $availableLanguages = [];
            }

            $episodesMap[$episodeId] = [
                'id' => $episodeId,
                'content_id' => (int)$row['content_id'],
                'episode_number' => (int)$row['episode_number'],
                'title' => $row['title'] ?? 'Episode ' . $row['episode_number'],
                'season' => (int)$row['season_number'],
                'season_id' => (int)$row['season_id'],
                'episode_language' => $row['episode_language'],
                'streams' => [],
                'languages' => $availableLanguages,
                'total_streams' => (int)$row['total_streams']
            ];
        }

        // Add stream if exists
        if ($row['stream_url'] !== null) {
            $episodesMap[$episodeId]['streams'][] = [
                'language' => $row['stream_language'],
                'hoster' => $row['hoster'],
                'stream_url' => $row['stream_url'],
                'quality' => $row['quality']
            ];
        }
    }

    $episodes = array_values($episodesMap);

    sendSuccess([
        'items' => $episodes,
        'count' => count($episodes),
        '_version' => '2024-10-17-OPCACHE-BUSTER'
    ]);

} catch (Exception $e) {
    error_log('[Episodes] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
