<?php
/**
 * Get Available Seasons for Content
 * GET /api/episodes/seasons?content_id=123
 * USES: mv_content_seasons (verified from VPS)
 * VERSION: 2024-10-17-OPCACHE-BUSTER
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

setCorsHeaders();
setCacheDynamic(360); // Cache for 6 hours
setJsonHeader();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Method not allowed', 405);
}

try {
    // Support both content_id and contentId for compatibility
    $contentId = (int)getParam('content_id', getParam('contentId', 0));

    if ($contentId <= 0) {
        sendError('Invalid content_id', 400);
    }

    // Use materialized view (verified from VPS)
    $sql = "
        SELECT
            season_number,
            episode_count,
            first_episode,
            last_episode
        FROM mv_content_seasons
        WHERE content_id = :content_id
        ORDER BY season_number ASC
    ";

    // Use PgBouncer for connection pooling (Materialized Views ARE accessible via PgBouncer)
    $results = dbQuery($sql, ['content_id' => $contentId]);

    if (empty($results)) {
        // Return default season if none found
        sendSuccess(['seasons' => [1]]);
        exit;
    }

    $seasons = array_map(function($row) {
        return (int)$row['season_number'];
    }, $results);

    sendSuccess([
        'seasons' => $seasons,
        'count' => count($seasons),
        '_version' => '2024-10-17-OPCACHE-BUSTER'
    ]);

} catch (Exception $e) {
    error_log('[Seasons] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
