<?php
/**
 * Seasons Endpoint - TIMING DEBUG
 * VERSION: 2024-10-17-TIMING-DEBUG
 */

function logTime($label) {
    global $timings;
    $timings[] = [
        'label' => $label,
        'time' => microtime(true),
        'memory' => memory_get_usage()
    ];
    error_log("[TIMING] $label: " . microtime(true));
}

$timings = [];
$startTime = microtime(true);
logTime('START');

require_once '../../../spam/db_config.php';
logTime('AFTER require db_config.php');

require_once '../../../spam/cache_config.php';
logTime('AFTER require cache_config.php');

require_once '../../../spam/helpers.php';
logTime('AFTER require helpers.php');

setCorsHeaders();
logTime('AFTER setCorsHeaders()');

setCacheDynamic(360);
logTime('AFTER setCacheDynamic()');

setJsonHeader();
logTime('AFTER setJsonHeader()');

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    sendError('Method not allowed', 405);
}
logTime('AFTER method check');

try {
    $contentId = (int)getParam('content_id', getParam('contentId', 123));
    logTime('AFTER getParam content_id');

    if ($contentId <= 0) {
        sendError('Invalid content_id', 400);
    }
    logTime('AFTER content_id validation');

    $sql = "
        SELECT
            season_number,
            episode_count,
            first_episode,
            last_episode
        FROM mv_content_seasons
        WHERE content_id = :content_id
        ORDER BY season_number ASC
    ";
    logTime('AFTER SQL string build');

    // ⏱️ CRITICAL: Measure getDbConnection time via dbQuery
    $beforeDbQuery = microtime(true);
    $results = dbQuery($sql, ['content_id' => $contentId]);
    $afterDbQuery = microtime(true);
    logTime('AFTER dbQuery() - TOTAL TIME: ' . round(($afterDbQuery - $beforeDbQuery) * 1000, 2) . 'ms');

    if (empty($results)) {
        sendSuccess(['seasons' => [1]]);
        exit;
    }
    logTime('AFTER empty check');

    $seasons = array_map(function($row) {
        return (int)$row['season_number'];
    }, $results);
    logTime('AFTER array_map');

    // Calculate detailed timings
    $processedTimings = [];
    for ($i = 1; $i < count($timings); $i++) {
        $prev = $timings[$i - 1];
        $curr = $timings[$i];
        $duration = ($curr['time'] - $prev['time']) * 1000;
        $processedTimings[] = [
            'step' => $curr['label'],
            'duration_ms' => round($duration, 3),
            'memory_mb' => round($curr['memory'] / 1024 / 1024, 2)
        ];
    }

    $totalTime = round((microtime(true) - $startTime) * 1000, 2);

    sendSuccess([
        'seasons' => $seasons,
        'count' => count($seasons),
        '_debug' => [
            'version' => '2024-10-17-TIMING-DEBUG',
            'total_time_ms' => $totalTime,
            'timings' => $processedTimings
        ]
    ]);

} catch (Exception $e) {
    logTime('EXCEPTION: ' . $e->getMessage());

    $processedTimings = [];
    for ($i = 1; $i < count($timings); $i++) {
        $prev = $timings[$i - 1];
        $curr = $timings[$i];
        $duration = ($curr['time'] - $prev['time']) * 1000;
        $processedTimings[] = [
            'step' => $curr['label'],
            'duration_ms' => round($duration, 3)
        ];
    }

    error_log('[Seasons Timing] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage() . ' | Timings: ' . json_encode($processedTimings), 500);
}
?>
