<?php
/**
 * Test actual query performance in episodes.php context
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

header('Content-Type: application/json');

$contentId = 4889;
$season = 1;

$timings = [];

// Test 1: DB Connection time
$start = microtime(true);
$pdo = getDbConnection(true); // Use PgBouncer
$timings['connection_time_ms'] = round((microtime(true) - $start) * 1000, 3);

// Test 2: Materialized View query (like seasons.php)
$start = microtime(true);
$sql = "
    SELECT season_number, episode_count, first_episode, last_episode
    FROM mv_content_seasons
    WHERE content_id = :content_id
    ORDER BY season_number ASC
";
$results = dbQuery($sql, ['content_id' => $contentId], true); // Use PgBouncer
$timings['seasons_query_ms'] = round((microtime(true) - $start) * 1000, 3);
$timings['seasons_count'] = count($results);

// Test 3: Episodes query (like episodes.php)
$start = microtime(true);
$sql = "
    SELECT
        mv.episode_id, mv.content_id, mv.episode_number, mv.title,
        mv.season_number, mv.season_id, mv.episode_language,
        mv.available_languages, mv.total_streams,
        st.language as stream_language, st.hoster, st.stream_url, st.quality
    FROM mv_episodes_with_streams mv
    LEFT JOIN streams st ON st.episode_id = mv.episode_id
    WHERE mv.content_id = :content_id AND mv.season_number = :season
    ORDER BY mv.season_number, mv.episode_number, st.language, st.hoster
";
$results = dbQuery($sql, ['content_id' => $contentId, 'season' => $season], true);
$timings['episodes_query_ms'] = round((microtime(true) - $start) * 1000, 3);
$timings['episodes_result_count'] = count($results);

// Test 4: Simple count query
$start = microtime(true);
$sql = "SELECT COUNT(*) FROM animes WHERE status = 'completed'";
$results = dbQuery($sql, [], true);
$timings['simple_count_ms'] = round((microtime(true) - $start) * 1000, 3);

// Calculate total time
$timings['total_test_time_ms'] = array_sum([
    $timings['connection_time_ms'],
    $timings['seasons_query_ms'],
    $timings['episodes_query_ms'],
    $timings['simple_count_ms']
]);

echo json_encode([
    'success' => true,
    'timings' => $timings,
    'using_pgbouncer' => true,
    'port' => DB_PORT_PGBOUNCER,
    'timestamp' => time()
], JSON_PRETTY_PRINT);
?>
