<?php
/**
 * Error Check & Debug Tool
 * URL: https://buy-business-manager.com/api/error-check.php
 */

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

header('Content-Type: application/json');

$checks = [
    'server_info' => [],
    'paths' => [],
    'extensions' => [],
    'database' => [],
    'files' => []
];

// 1. Server Info
$checks['server_info'] = [
    'php_version' => phpversion(),
    'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
    'document_root' => $_SERVER['DOCUMENT_ROOT'] ?? 'Unknown',
    'script_filename' => __FILE__,
    'current_dir' => __DIR__,
    'user' => get_current_user(),
];

// 2. Path Checks
// FROM: /home/thnojfxl/public_html/api/error-check.php
// __DIR__ = /home/thnojfxl/public_html/api
// dirname(__DIR__, 1) = /home/thnojfxl/public_html
// dirname(__DIR__, 2) = /home/thnojfxl ✅
$baseDir = dirname(__DIR__, 2); // /home/thnojfxl
$checks['paths'] = [
    'base_dir' => $baseDir,
    'spam_dir' => $baseDir . '/spam',
    'spam_exists' => is_dir($baseDir . '/spam'),
    'spam_readable' => is_readable($baseDir . '/spam'),
    'public_html' => $baseDir . '/public_html',
];

// 3. Required PHP Extensions
$requiredExt = ['pdo', 'pdo_pgsql', 'pgsql', 'mbstring', 'json', 'curl'];
foreach ($requiredExt as $ext) {
    $checks['extensions'][$ext] = extension_loaded($ext);
}

// 4. Config Files
$configFiles = [
    'db_config.php' => $baseDir . '/spam/db_config.php',
    'cache_config.php' => $baseDir . '/spam/cache_config.php',
    'helpers.php' => $baseDir . '/spam/helpers.php',
];

foreach ($configFiles as $name => $path) {
    $checks['files'][$name] = [
        'path' => $path,
        'exists' => file_exists($path),
        'readable' => is_readable($path),
        'size' => file_exists($path) ? filesize($path) : 0,
    ];
}

// 5. Try to load db_config.php
$dbConfigPath = $baseDir . '/spam/db_config.php';
$checks['database']['config_load'] = false;
$checks['database']['error'] = null;

if (file_exists($dbConfigPath)) {
    try {
        require_once $dbConfigPath;
        $checks['database']['config_load'] = true;

        // Check if constants are defined
        $checks['database']['constants'] = [
            'DB_HOST' => defined('DB_HOST') ? DB_HOST : 'NOT DEFINED',
            'DB_NAME' => defined('DB_NAME') ? DB_NAME : 'NOT DEFINED',
            'DB_USER' => defined('DB_USER') ? DB_USER : 'NOT DEFINED',
            'DB_PORT_PGBOUNCER' => defined('DB_PORT_PGBOUNCER') ? DB_PORT_PGBOUNCER : 'NOT DEFINED',
        ];

        // Try to connect
        if (function_exists('getDbConnection')) {
            try {
                $pdo = getDbConnection(true);
                $checks['database']['connection'] = 'SUCCESS';

                // Test query
                $stmt = $pdo->query('SELECT 1 as test');
                $result = $stmt->fetch();
                $checks['database']['test_query'] = $result ? 'SUCCESS' : 'FAILED';
            } catch (Exception $e) {
                $checks['database']['connection'] = 'FAILED';
                $checks['database']['connection_error'] = $e->getMessage();
            }
        }
    } catch (Exception $e) {
        $checks['database']['error'] = $e->getMessage();
    }
}

// 6. Check OpCache
$checks['opcache'] = [
    'enabled' => function_exists('opcache_get_status'),
    'status' => function_exists('opcache_get_status') ? opcache_get_status(false) : null,
];

echo json_encode($checks, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
?>
