<?php
/**
 * Submit Content Rating
 * POST /api/ratings/submit
 *
 * Request Body:
 * {
 *   "user_id": 1,
 *   "content_id": 123,
 *   "rating": 5
 * }
 *
 * Note: content_id maps to anime_id in the database
 * Rating scale: 1-5 stars (not 1-10!)
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

setCorsHeaders();
setCacheNone(); // Don't cache rating submissions
setJsonHeader();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendError('Method not allowed', 405);
}

try {
    $data = getJsonBody();

    // Validate required parameters
    $validation = validateParams(['user_id', 'content_id', 'rating'], $data);
    if ($validation !== true) {
        sendError($validation, 400);
    }

    $userId = (int)$data['user_id'];
    $contentId = (int)$data['content_id'];
    $rating = (int)$data['rating'];

    // Validate values
    if ($userId <= 0 || $contentId <= 0) {
        sendError('Invalid user_id or content_id', 400);
    }

    if ($rating < 1 || $rating > 5) {
        sendError('Rating must be between 1 and 5', 400);
    }

    // Check if rating already exists (UPSERT) - using anime_id
    $checkSql = "SELECT id FROM user_ratings WHERE user_id = :user_id AND anime_id = :anime_id";
    $existing = dbQuery($checkSql, [
        'user_id' => $userId,
        'anime_id' => $contentId
    ]);

    if (empty($existing)) {
        // Insert new rating
        $insertSql = "
            INSERT INTO user_ratings (user_id, anime_id, rating, created_at)
            VALUES (:user_id, :anime_id, :rating, CURRENT_TIMESTAMP)
        ";
        dbExecute($insertSql, [
            'user_id' => $userId,
            'anime_id' => $contentId,
            'rating' => $rating
        ]);
    } else {
        // Update existing rating
        $updateSql = "
            UPDATE user_ratings
            SET rating = :rating, updated_at = CURRENT_TIMESTAMP
            WHERE user_id = :user_id AND anime_id = :anime_id
        ";
        dbExecute($updateSql, [
            'user_id' => $userId,
            'anime_id' => $contentId,
            'rating' => $rating
        ]);
    }

    // Update average rating in animes table
    try {
        $avgSql = "
            UPDATE animes
            SET rating = (
                SELECT AVG(rating)::NUMERIC(3,1)
                FROM user_ratings
                WHERE anime_id = :anime_id
            )
            WHERE id = :anime_id
        ";
        dbExecute($avgSql, ['anime_id' => $contentId]);
    } catch (Exception $e) {
        // Ignore if animes table doesn't have rating column or error
        error_log('[SubmitRating] Could not update average rating: ' . $e->getMessage());
    }

    sendSuccess([
        'rating' => $rating,
        'user_id' => $userId,
        'content_id' => $contentId
    ], 'Rating saved successfully');

} catch (Exception $e) {
    error_log('[SubmitRating] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
