<?php
/**
 * User Favorites Endpoint
 * GET /api/user/favorites?user_id=123 - Get favorites
 * POST /api/user/favorites - Add favorite (body: user_id, content_id)
 * DELETE /api/user/favorites?user_id=123&content_id=456 - Remove favorite
 */

require_once '../../../spam/db_config.php';
require_once '../../../spam/cache_config.php';
require_once '../../../spam/helpers.php';

setCorsHeaders();
setCacheNone(); // User-specific data
setJsonHeader();

try {
    $method = $_SERVER['REQUEST_METHOD'];

    if ($method === 'GET') {
        // Get favorites
        $userId = (int)getParam('user_id');

        if ($userId <= 0) {
            sendError('Invalid user_id', 400);
        }

        $sql = "
            SELECT f.id, f.anime_id as content_id, f.added_at,
                   a.title, a.cover_url, a.rating, a.source_site
            FROM user_favorites f
            JOIN animes a ON a.id = f.anime_id
            WHERE f.user_id = :user_id
            ORDER BY f.added_at DESC
        ";

        $results = dbQuery($sql, ['user_id' => $userId]);

        $favorites = array_map(function($row) {
            return [
                'id' => (int)$row['id'],
                'content_id' => (int)$row['content_id'],
                'added_at' => $row['added_at'],
                'content' => [
                    'title' => $row['title'],
                    'cover_url' => $row['cover_url'],
                    'rating' => $row['rating'] ? (float)$row['rating'] : null,
                    'source_site' => $row['source_site']
                ]
            ];
        }, $results);

        sendSuccess(['items' => $favorites, 'count' => count($favorites)]);

    } elseif ($method === 'POST') {
        // Add favorite
        $data = getJsonBody();
        $validation = validateParams(['user_id', 'content_id'], $data);

        if ($validation !== true) {
            sendError($validation, 400);
        }

        $sql = "
            INSERT INTO user_favorites (user_id, anime_id, added_at)
            VALUES (:user_id, :content_id, CURRENT_TIMESTAMP)
            ON CONFLICT (user_id, anime_id) DO NOTHING
            RETURNING id
        ";

        $result = dbQuery($sql, [
            'user_id' => (int)$data['user_id'],
            'content_id' => (int)$data['content_id']
        ]);

        sendSuccess(['added' => !empty($result)], 'Added to favorites');

    } elseif ($method === 'DELETE') {
        // Remove favorite
        $userId = (int)getParam('user_id');
        $contentId = (int)getParam('content_id');

        if ($userId <= 0 || $contentId <= 0) {
            sendError('Invalid user_id or content_id', 400);
        }

        $sql = "
            DELETE FROM user_favorites
            WHERE user_id = :user_id AND anime_id = :content_id
        ";

        $rowCount = dbExecute($sql, [
            'user_id' => $userId,
            'content_id' => $contentId
        ]);

        sendSuccess(['removed' => $rowCount > 0], 'Removed from favorites');

    } else {
        sendError('Method not allowed', 405);
    }

} catch (Exception $e) {
    error_log('[Favorites] Error: ' . $e->getMessage());
    sendError('Server error: ' . $e->getMessage(), 500);
}
?>
